<?php


/**
 * @todo : Related shortcode functions
 */

if ( ! class_exists( 'ST_Pb_Helper_Shortcode' ) ) {

	class ST_Pb_Helper_Shortcode {

		static private $debug = true;

		static $pattern = '';
		static $group_shortcodes = array( 'group', 'group_table', 'table' );
		static $item_html_template = array(
			'icon' => "<i class='_ST_STD_'></i>",
		);
		static $notice = array();

		/**
		 * Get what tab (Classic - Pagebuilder) is active when Save content of given post
		 * and replaces $content with tab content if available
		 * @param  string  $content            given post content from classic tab
		 * @param  object  $post               wp post object
		 * @param  bool  $st_page_active_tab can be given to override check
		 * @param  [type]  $st_deactivate_pb can be given to override check
		 * @param  boolean $allow_show_post    can be given to override check
		 * @return string                      final content, either unreplaced from classic editor or replaced with builder tab content
		 */
		static public function get_builder_tab_content($content, $post, $st_page_active_tab = null, $st_deactivate_pb = null, $allow_show_post = false, $process=true)
		{

			if(! $st_page_active_tab)
				$st_page_active_tab = get_post_meta( $post->ID, '_wr_page_active_tab', true );

			if(! $st_deactivate_pb)
				$st_deactivate_pb = get_post_meta( $post->ID, '_wr_deactivate_pb', true );

			if(! $allow_show_post)
			{
				// Check password protected in post
				$allow_show_post = false;
				if ( 'publish' == $post->post_status && empty( $post->post_password ) ) {
					$allow_show_post = true;
				}
			}

			// if Pagebuilder is active when save and pagebuilder is not deactivate on this post
			if ( $st_page_active_tab && empty( $st_deactivate_pb ) && $allow_show_post == true ) {
				$st_pagebuilder_content = get_post_meta( $post->ID, '_wr_page_builder_content', true );
				if ( ! empty( $st_pagebuilder_content ) )
				{
					$content = self::_process_builder_tab_content($content, $process);
					$content = $st_pagebuilder_content;
				}
			}

			return $content;
		}

		static public function _process_builder_tab_content($content, $process)
		{
			if($process == false) return $content;

			// remove placeholder text which was inserted to &lt; and &gt;
			$content = ST_Pb_Utils_Placeholder::remove_placeholder( $content, 'wrapper_append', '' );

			$content = preg_replace_callback(
					'/\[st_widget\s+([A-Za-z0-9_-]+=\"[^"\']*\"\s*)*\s*\](.*)\[\/st_widget\]/Us', array( 'ST_Pb_Helper_Shortcode', 'widget_content' ), $content
			);

			return $content;

		}


		/**
		 * Reload ig shortcodes data and store in database, for next time
		 */
		public static function reload_wr_shortcodes( $force_reload = 0 ) {

			// yunus edit - search this file for other references to _wr_pb_providers_v
			$providers = unserialize( get_option( '_ST_defualt_providers_v2' ) );

			if ( ! $force_reload ) {

				if ( ! $providers ) {
					self::reload_wr_shortcodes( 1 );
				}

				foreach ( $providers as $dir => $provider ) {
					$is_not_active = isset( $provider['file'] ) && ! is_plugin_active( $provider['file'] ) && $provider['name'] != __( 'Standard Elements', ST_PBL );

					// if a provider is removed
					if ( ! file_exists( $dir ) || $is_not_active ) {
						self::reload_wr_shortcodes( 1 );
						return ;
					}
				}
			}

			// Do action to register addon providers
			do_action( 'st_pb_addon' );

			// Get list of providers
			global $St_Sc_Providers;

			$St_Sc_Providers = apply_filters(
				'st_pb_provider',
				self::register_provider()
			);

			//echo '<pre>'.print_r($providers, TRUE).'</pre>';

			// Compare current provides vs stored providers
			// yunus edit - providers check, not really edit, just a note
			if ( $providers == $St_Sc_Providers ) {
				return ;
			}

			// Update providers data
			update_option( '_ST_defualt_providers_v2', serialize( $St_Sc_Providers ) );

			// compatibility check
			self::compatibility_check();

			// show notice
			add_action( 'admin_notices', array( __CLASS__, 'show_notice' ), 100 );

			$sc_autoload_register = array();
			// Get list of shortcode directories
			$sc_path = self::shortcode_dirs();

			foreach ( $sc_path as $path ) {
				if ( ! isset( $sc_autoload_register['path'] ) ) {
					$sc_autoload_register['path'] = array();
				}
				$sc_autoload_register['path'][] = $path;

				// List files and directories in shortcodes directory
				$excludes = array( '.', '..' );
				$dir_entries = scandir( $path );
				foreach ( $dir_entries as $entry ) {

					// Add folder of each shortcode to autoload list
					if ( is_dir( $path . '/' . $entry ) && ! in_array( $entry, $excludes ) ) {
						if ( ! isset( $sc_autoload_register['register'] ) ) {
							$sc_autoload_register['register'] = array();
						}
						$sc_autoload_register['register'][] = $path . '/' . $entry;
					}
				}
			}

			// Update shortcodes data
			update_option( '_st_pb_autoload_register', serialize( $sc_autoload_register ) );

			// Get list of shortcodes
			$shortcodes = self::shortcodes_list( $sc_path );

			// Update shortcodes data
			update_option( '_st_pb_shortcodes', serialize( $shortcodes ) );

			// Reset reload option
			update_option( '_st_pb_reload_shortcodes', 0 );
		}

		/**
		 * Get list of (element + layout) shortcodes from shortcode folders
		 * @return array
		 */
		public static function st_pb_shortcode_tags() {

			// Get list of providers
			global $St_Sc_Providers, $St_Sc_By_Providers, $St_Sc_By_Providers_Name;

			self::reload_wr_shortcodes();

			// Get stored data
			$St_Sc_Providers = unserialize( get_option( '_ST_defualt_providers_v2' ) );
			$shortcodes = unserialize( get_option( '_st_pb_shortcodes' ) );
			$sc_autoload_register = unserialize( get_option( '_st_pb_autoload_register' ) );

			$St_Sc_By_Providers = unserialize( get_option( '_wr_pb_sc_by_providers' ) );
			$St_Sc_By_Providers_Name = unserialize( get_option( '_wr_pb_sc_by_providers_name' ) );

			foreach ( (array) $sc_autoload_register['path'] as $path ) {
				self::autoload_shortcodes( $path );
			}

			foreach ( (array) $sc_autoload_register['register'] as $path_entry ) {
				ST_Pb_Loader::register( $path_entry , 'ST_' );
			}

			// ------------------------------------------------------
			//  yunus edit - duplicates layout elements to 4 levels for nesting
			// ------------------------------------------------------
			$shortcodes_clone = $shortcodes;
			$shortcodes = array();
			if($shortcodes_clone && is_array($shortcodes_clone) ) foreach($shortcodes_clone as $name=>$config)
			{
				$shortcodes[ $name ] = $config;
				if(isset($config['type']) && $config['type'] == 'layout')
				{
					$shortcodes[ $name . '_2' ] = $config;
					$shortcodes[ $name . '_3' ] = $config;
					$shortcodes[ $name . '_4' ] = $config;
				}
			}

			return $shortcodes;
		}

		/**
		 * Check compatibility of Addons vs Core
		 *
		 * @global type $St_Sc_Providers
		 */
		public static function compatibility_check() {

			global $St_Sc_Providers;
			$providers = $St_Sc_Providers;

			// get current version of core
			$core_version = ST_Pb_Helper_Functions::get_plugin_info( ST_PB_FILE, 'Version' );

			foreach ( $providers as $dir => $provider ) {
				if ( ! empty ( $provider['file'] ) && ! empty ( $provider['path'] ) ) {

					$addon_file = $provider['file'];

					// get value of core version required
					$core_required = ST_Pb_Addon::core_version_requied_value( $provider, $addon_file );

					if ( $core_required ) {
						// addon plugin name
						$addon_name = ST_Pb_Helper_Functions::get_plugin_info( $provider['file_path'], 'Name' );

						$compatibility = ST_Pb_Addon::compatibility_handle( $core_required, $core_version, $addon_file );

						if ( ! $compatibility ) {
							// remove provider from list
							unset ( $St_Sc_Providers[$dir] );

							// show notice
							self::$notice[] = ST_Pb_Addon::show_notice( array( 'addon_name' => $addon_name, 'core_required' => $core_required ), 'core_required' );
						}
					}
				}
			}
		}

		/**
		 * Show notice on top of admin pages
		 */
		public static function show_notice() {
			if ( self::$notice ) {
				echo balanceTags( implode( '', self::$notice ) );
			}
		}

		/**
		 * Autoload shortcodes & sub shortcodes
		 *
		 * @param string $path
		 */
		public static function autoload_shortcodes( $path ) {
			$items   = substr_count( $path, '/item' );
			$postfix = str_repeat( 'Item_', $items );
			// autoload shortcodes
			ST_Pb_Loader::register( $path, 'ST_' . $postfix );
		}

		/**
		 * Set information for WooRockets provider
		 *
		 * @return type
		 */
		public static function register_provider() {
			return array(
				plugin_dir_path( ST_PB_FILE ) =>
				array(
					'path'             => ST_PB_PATH,
					'uri'              => ST_PB_URI,
					'name'             => 'Layout',
					'shortcode_dir'    => array( ST_PB_LAYOUT_PATH ), //array( ST_PB_LAYOUT_PATH, ST_PB_ELEMENT_PATH ),
					'js_shortcode_dir' => array(
						'path' => ST_PB_PATH . 'assets/main/js/shortcodes',
						'uri'  => ST_PB_URI . 'assets/main/js/shortcodes',
					),
				)
			);
		}

		/**
		 * Get provider name & path of a shortcode directory
		 *
		 * @param type $shortcode_dir
		 *
		 * @return type
		 */
		public static function get_provider( $shortcode_dir ) {
			global $St_Sc_Providers;
			$providers = $St_Sc_Providers;
			foreach ( $providers as $dir => $provider ) {
				foreach ( (array) $provider['shortcode_dir'] as $dir ) {
					if ( strpos( $shortcode_dir, $dir ) !== false ) {
						return array(
							'name' => $provider['name'],
							'dir'  => $dir,
						);
					}
				}
			}
		}

		/**
		 * Get info of provider of the shortcode
		 *
		 * @global type $St_Sc_Providers , $St_Sc_By_Providers
		 *
		 * @param type  $shortcode_name
		 * @param type  $shortcode_by_providers
		 *
		 * @return type
		 */
		public static function get_provider_info( $shortcode_name, $info ) {
			global $St_Sc_Providers;
			global $St_Sc_By_Providers;
			$providers              = $St_Sc_Providers;
			$shortcode_by_providers = $St_Sc_By_Providers;
			foreach ( $shortcode_by_providers as $provider_dir => $shortcodes ) {
				// find shortcode in what directory
				if ( in_array( $shortcode_name, $shortcodes ) ) {
					// find provider of that directory
					foreach ( $providers as $dir => $provider ) {
						foreach ( (array) $provider['shortcode_dir'] as $dir ) {
							if ( $provider_dir == $dir ) {
								return $St_Sc_Providers[$provider['path']][$info];
							}
						}
					}
				}
			}
		}

		/**
		 * Get shortcode directories of providers
		 *
		 * @return type
		 */
		public static function shortcode_dirs() {
			global $St_Sc_Providers;
			$providers      = $St_Sc_Providers;
			$shortcode_dirs = array();
			foreach ( $providers as $provider ) {
				$shortcode_dirs = array_merge( $shortcode_dirs, (array) $provider['shortcode_dir'] );
			}

			return $shortcode_dirs;
		}

		/**
		 * Get shortcodes in shortcode directories
		 *
		 * @param array $sc_path
		 *
		 * @return array
		 */
		public static function shortcodes_list( $sc_path ) {
			if ( empty( $sc_path ) ) {
				return NULL;
			}
			if ( ! is_array( $sc_path ) ) {
				$sc_path = array( $sc_path );
			}
			// array of shortcodes by shortcode path
			global $St_Sc_By_Providers;

			// array of shortcodes by provider name
			global $St_Sc_By_Providers_Name;

			// get list of directory by directory level
			$level_dirs = array();
			foreach ( $sc_path as $path ) {
				$level_dirs[substr_count( $path, '/*' )][] = $path;

				// List files and directories in shortcodes directory
				$excludes = array( '.', '..' );
				$dir_entries = scandir( $path );
				foreach ( $dir_entries as $entry ) {

					// path to a shortcode directory
					$shortcode_path = $path . '/' . $entry;

					if ( is_dir( $shortcode_path ) && ! in_array( $entry, $excludes ) ) {

						// Add path to shortcode folder to 1st level array
						$level_dirs[1][] = $shortcode_path;

						// check if exist /item folder (contains sub shortcodes), add it to 2rd level array
						if ( is_dir( $shortcode_path . '/item' ) ) {
							$level_dirs[2][] = $shortcode_path . '/item';
						}
					}
				}
			}

			// store all shortcodes
			$shortcodes = array();

			// traverse over array of path to get shortcode information
			foreach ( $level_dirs as $level => $dirs ) {
				foreach ( $dirs as $dir ) {
					// provider info
					$parent_path = str_replace( '/item', '', $dir );
					$provider    = self::get_provider( $parent_path );
					// shortcode info
					$type       = ( $dir == ST_PB_LAYOUT_PATH ) ? 'layout' : 'element';
					$this_level = ( intval( $level ) > 0 ) ? ( intval( $level ) - 1 ) : intval( $level );
					$append     = str_repeat( 'item_', $this_level );

					// get file .php from directory of each shortcode
					if ( count( (array) glob( $dir . '/*.php' ) ) > 0 ) {
						foreach ( glob( $dir . '/*.php' ) as $file ) {
							// Skip including main initialization file
							if ( 'main.php' != substr( $file, - 8 ) ) {
								$p                           = pathinfo( $file );
								$element                     = str_replace( '-', '_', $p['filename'] );
								$shortcode_name              = 'st_' . $append . $element;
								$shortcodes[$shortcode_name] = array( 'type' => $type, 'provider' => $provider );

								$St_Sc_By_Providers[$provider['dir']][]       = $shortcode_name;
								$St_Sc_By_Providers_Name[$provider['name']][] = $shortcode_name;
							}
						}
					}
				}
			}

			// Add/Update option
			update_option( '_wr_pb_sc_by_providers', serialize( $St_Sc_By_Providers ) );
			update_option( '_wr_pb_sc_by_providers_name', serialize( $St_Sc_By_Providers_Name ) );

			return $shortcodes;
		}

		/**
		 * Extract shortcode params from string
		 * Ex: [param-tag=h3&param-text=Your+heading+text&param-font=custom]
		 *
		 * @param type $param_str
		 *
		 * @return array
		 */
		static function extract_params( $param_str, $str_shortcode = '' ) {

			if(is_array($param_str)) return $param_str;

			$param_str = stripslashes( $param_str );

			// Get shortcode name & parameters
			preg_match_all( '/\[[^\s"]+\s+([A-Za-z0-9_-]+=\"[^"]*\"\s*)*\s*\]/', $param_str, $rg_sc_params );
			if ( empty( $rg_sc_params[0] ) ) {
				return '';
			}
			$sc_name_params = ! empty( $rg_sc_params[0][0] ) ? $rg_sc_params[0][0] : $rg_sc_params[0];


			$params    = array();
			// get params of shortcode
			preg_match_all( '/[A-Za-z0-9_-]+=\"[^"]*\"/u', $sc_name_params, $tmp_params, PREG_PATTERN_ORDER );

			foreach ( $tmp_params[0] as $param_value ) {
				$output = array();
				preg_match_all( '/([A-Za-z0-9_-]+)=\"([^"]*)\"/u', $param_value, $output, PREG_SET_ORDER );
				foreach ( $output as $item ) {
					if ( ! in_array( $item[1], array( 'disabled_el', 'css_suffix' ) ) || ! isset ( $params[$item[1]] ) ) {
						$params[$item[1]] = urldecode( $item[2] );
					}
				}
			}
			$pattern = get_shortcode_regex();
			preg_match_all( '/' . $pattern . '/s', $param_str, $tmp_params, PREG_PATTERN_ORDER );
			$content                      = isset( $tmp_params[5][0] ) ? trim( $tmp_params[5][0] ) : '';
			$content                      = preg_replace( '/rich_content_param-[a-z_]+=/', '', $content );
			$params['_shortcode_content'] = $content;

			return $params;
		}

		/**
		 * Join params to shortcode structure string
		 *
		 * @param array  $params
		 * @param string $shortcode_name
		 * @param string $content string between shortcode
		 *                        open and close tags
		 *
		 * @return string
		 */
		static function join_params( $params, $shortcode_name, $content = '' ) {
			$shortcode_structure = '[' . $shortcode_name;
			if ( is_array( $params ) && count( $params ) ) {
				foreach ( $params as $k => $param ) {
					$shortcode_structure .= ' ' . $k . '="' . $param . '"';
				}
			}
			$shortcode_structure .= ']';
			$shortcode_structure .= $content;
			$shortcode_structure .= '[/' . $shortcode_name . ']';

			return $shortcode_structure;
		}

		/**
		 * Generate options list of shortcode (from $this->items array) OR get value of a option
		 *
		 * @param array       $arr             ($this->items)
		 * @param string|null $paramID         (get std of a option by ID)
		 * @param array       $new_values      (set std for some options ( "pram id" => "new std value" ))
		 * @param bool        $assign_content  (set $option['std'] = $new_values['_shortcode_content'] of option which has role = 'content')
		 * @param bool        $extract_content (get $option['std'] of option which has role = 'content')
		 * @param string      $extract_title   (get $option['std'] of option which has role|role_2 = 'title')
		 *
		 * @return array
		 */
		static function generate_shortcode_params( &$arr, $paramID = NULL, $new_values = NULL, $assign_content = FALSE, $extract_content = FALSE, $extract_title = '', &$has_preview = false, &$preview_config=array('size'=>'default') ) {
			$params = array();


			// ------------------------------------------------------
			//  MAIN ELEMENT ID
			// ------------------------------------------------------
			$params['elm_id'] = 'elm_'.uniqid();

			if ( ! $arr )
			{
				return $params;
			}

			foreach ( $arr as $tab => &$options ) // for each tabs
			{
				foreach ( $options as &$option ) // for each field in tab
				{
					$type          = isset( $option['type'] ) ? $option['type'] : '';
					$option['std'] = ! isset( $option['std'] ) ? '' : $option['std'];

					// option has role = 'content'
					if ( isset( $option['role'] ) && $option['role'] == 'content' ) {

						// set std of this option
						if ( $assign_content ) {
							if ( ! empty( $new_values ) && isset( $new_values['_shortcode_content'] ) ) {
								$option['std'] = $new_values['_shortcode_content'];
							}
						}

						// get std of this option
						if ( $extract_content ) {
							$params['extract_shortcode_content'][$option['id']] = $option['std'];
						} else {
							// remove option which role = content from shortcode structure ( except option which has another role: title )
							if ( ! (
								( isset( $option['role'] ) && $option['role'] == 'title' ) ||
								( isset( $option['role_2'] ) && $option['role_2'] == 'title' )
								|| ( isset( $option['role'] ) && $option['role'] == 'title_prepend' ) ) )
							{
								unset( $option );
								continue;
							}
						}
					}

					if ( $type == 'preview' )
					{
						// ------------------------------------------------------
						//  yunus edit - removed preview pane fron returned form as we are going to handle it differently on its own modal window
						// ------------------------------------------------------
						$has_preview = true;
						$preview_config['size'] = isset( $option['size'] ) ? $option['size'] : $preview_config['size'];

						if(! isset($preview_config['bg'])) $preview_config['bg'] = '';
						$preview_config['bg'] = isset( $option['bg'] ) ? $option['bg'] : $preview_config['bg'];

						continue;
					}

					// single option : $option['type'] => string
					if ( ! is_array( $type ) )
					{
						// if is not parent/nested shortcode
						if ( ! in_array( $type, self::$group_shortcodes ) )
						{

							// default content
							if ( empty( $new_values ) ) {
								if ( ! empty( $paramID ) ) {
									if ( $option['id'] == $paramID ) {
										return $option['std'];
									}
								} else {
									if ( isset( $option['id'] ) ) {
										$params[$option['id']] = $option['std'];
									}
								}
							} // there are new values
							else {
								if ( isset( $option['id'] ) && array_key_exists( $option['id'], $new_values ) ) {
									$option['std'] = $new_values[$option['id']];
								}
								if ( isset( $option['role'] ) && $option['role'] == 'title' && empty( $new_values[$option['id']] ) ) {
									//$option['std'] = '';
								}
							}

							// extract title for element like Table
							if ( ! empty( $extract_title ) ) {
								// default std
								if ( strpos( $option['std'], ST_Pb_Utils_Placeholder::get_placeholder( 'index' ) ) !== false ) {
									$option['std']           = $extract_title;
									$params['extract_title'] = $extract_title;
								} else {
									if ( ( isset( $option['role'] ) && $option['role'] == 'title' ) || ( isset( $option['role_2'] ) && $option['role_2'] == 'title' ) ) {
										if ( $option['role'] == 'title' ) {
											$params['extract_title'] = $option['std'];
										} else {
											$params['extract_title'] = ST_Pb_Utils_Common::slice_content( $option['std'] );
										}
									} else {
										if ( ( isset( $option['role'] ) && $option['role'] == 'title_prepend' ) && ! empty( $option['title_prepend_type'] ) && ! empty( $option['std'] ) ) {
											$params['extract_title'] = ST_Pb_Utils_Placeholder::remove_placeholder( self::$item_html_template[$option['title_prepend_type']], 'standard_value', $option['std'] ) . $params['extract_title'];
										}
									}
								}
							}
						} // nested shortcode
						else
						{
							// default content
							if ( empty( $new_values ) ) {
								foreach ( $option['sub_items'] as &$sub_items ) {
									$sub_items['std'] = ! isset( $sub_items['std'] ) ? '' : $sub_items['std'];

									if ( ! empty( $paramID ) ) {
										if ( $sub_items['id'] == $paramID ) {
											return $sub_items['std'];
										}
									} else {
										$params['sub_items_content'][$option['sub_item_type']][] = $sub_items;
									}
								}
							} // there are new values
							else {
								$count_default = count( $option['sub_items'] );
								$count_real    = isset( $new_values['sub_items_content'][$option['sub_item_type']] ) ? count( $new_values['sub_items_content'][$option['sub_item_type']] ) : 0;
								if ( $count_real > 0 ) {
									// there are new sub items
									if ( $count_default < $count_real ) {
										for ( $index = $count_default; $index < $count_real; $index ++ ) {
											$option['sub_items'][$index] = array( 'std' => '' );
										}
									} // some sub items are deleted
									else {
										if ( $count_default > $count_real ) {
											for ( $index = $count_real; $index < $count_default; $index ++ ) {
												unset( $option['sub_items'][$index] );
											}
										}
									}

									// update content for sub items
									array_walk( $option['sub_items'], array( 'ST_Pb_Helper_Functions', 'st_arr_walk_subsc' ), $new_values['sub_items_content'][$option['sub_item_type']] );
								}
							}
						}
					}
					else  // nested options : $option['type'] => Array of options
					{
						// default content
						if ( empty( $new_values ) )
						{
							foreach ( $option['type'] as &$sub_options ) {
								$sub_options['std'] = ! isset( $sub_options['std'] ) ? '' : $sub_options['std'];

								if ( ! empty( $paramID ) ) {
									if ( $sub_options['id'] == $paramID ) {
										return $sub_options['std'];
									}
								} else {
									$params[$sub_options['id']] = $sub_options['std'];
								}
							}
						}
						else  // there are new values
						{
							array_walk( $option['type'], array( 'ST_Pb_Helper_Functions', 'st_arr_walk' ), $new_values );
						}
					} // END ELSE OF single option : $option['type'] => string

					if ( isset( $option['extended_ids'] ) ) {
						foreach ( $option['extended_ids'] as $_id ) {
							$params[$_id] = isset( $option[$_id]['std'] ) ? $option[$_id]['std'] : '';
						}
					}

				} // END for each field in tab
			} // END for each tabs


			// if(! trim($params['id_wrapper']))
			// {
				//$params['id_wrapper'] = 'elm_'.uniqid();
			//}
			//echo '<pre>'.print_r($params, TRUE).'</pre>';

			return $params;
		}

		/**
		 * Generate shortcode structure from array of params and name of shortcode
		 *
		 * @param type $shortcode_name
		 * @param type $params
		 *
		 * @return type
		 */
		static function generate_shortcode_structure( $shortcode_name, $params, $content = '' ) {
			$shortcode_structure = "[$shortcode_name";

			$params = self::check_and_assign_id($params);

			$arr            = array();
			$exclude_params = array( 'sub_items_content', 'extract_shortcode_content' );
			if(! empty($params))
			{
				foreach ( $params as $key => $value ) {
					if ( ! in_array( $key, $exclude_params ) && $key != '' ) {
						$arr[$key] = $value;
					}
				}
			}

			// get content of param which has: role = content
			if ( isset($params['extract_shortcode_content']) && ! empty( $params['extract_shortcode_content'] ) ) {
				foreach ( $params['extract_shortcode_content'] as $paramId => $std ) {
					unset( $arr[$paramId] );
					$content = $std;
				}
			}

			if(! empty($arr))
			{
				$shortcode_structure .= ' ';
				foreach ( $arr as $key => $value ) {
					$shortcode_structure .= "$key=\"$value\" ";
				}
			}

			$shortcode_structure .= ']';
			$shortcode_structure .= $content;
			$shortcode_structure .= "[/$shortcode_name]";

			return $shortcode_structure;
		}

		/**
		 * Get Shortcode class from shortcode name
		 *
		 * @param type $shortcode_name
		 *
		 * @return type
		 */
		static function get_shortcode_class( $shortcode_name ) {
			$shortcode_name = str_replace( 'st_', 'ST_', $shortcode_name );
			$shortcode      = str_replace( '_', ' ', $shortcode_name );
			$class          = ucwords( $shortcode );
			$class          = str_replace( ' ', '_', $class );

			// for nesting layouts return original class name (without _[number])
			if(preg_match('/st_(row|column|subrow)_.+/i', $class)) $class = substr($class, 0, -2);

			return $class;
		}

		/**
		 * Return shortcode name without 'st_' prefix
		 *
		 * @param type $st_shortcode_name
		 *
		 * @return type
		 */
		static function shortcode_name( $st_shortcode_name ) {
			return str_replace( 'st_', '', $st_shortcode_name );
		}

		/**
		 * Removes wordpress autop and invalid nesting of p tags, as well as br tags
		 *
		 * @param string $content html content by the wordpress editor
		 * @param bool $autop Whether or not use wpautop
		 *
		 * @return string $content
		 */
		static function remove_autop( $content, $autop = true ) {
			$shortcode_tags = array();
			$tagregexp      = join( '|', array_map( 'preg_quote', $shortcode_tags ) );

			// find ig shortcode, initialize shortcode class
			if ( is_admin() ) {
				// ig shortcode list
				$tagregexp = self::regex_wr_sc_tag();

				// find ig shortcode
				$regex   = "\\[\\/($tagregexp)\\]";
				preg_match( "/$regex/s", $content, $matches, PREG_OFFSET_CAPTURE );
				if( $matches ) {
					// get shortcode tag
					$matched_sc = $matches[1][0] | '';
					if ( ! empty( $matched_sc ) ) {
						// get shortcode class name
						$sc_class = self::get_shortcode_class( $matched_sc );

						// initialize class
						if ( class_exists( $sc_class ) ) {
							$sc_object = new $sc_class();
						}
					}
				}
			}

			// yunus edit - uncommented this as i am not sure why even they have this in place
/*
			if ( defined( 'WP_ADMIN' ) ) {
				// Keep non-WR shortcode as they are in preview iframe
				$content = str_replace( '[', '&#91;', $content );
				$content = str_replace( '&#91;st_', '[st_', $content );
				$content = str_replace( '&#91;/st_', '[/st_', $content );
			}
*/

			// yunus edit - support for oembed
			if(isset($GLOBALS['wp_embed']))
			{
				$content = $GLOBALS['wp_embed']->run_shortcode($content);
				$content = $GLOBALS['wp_embed']->autoembed($content);
			}

			$content = do_shortcode( $content );

			// if myCred plugin is activated then use the temporary workaround (2/2)
			// remove our advanced autop filter
			if( is_plugin_active( 'mycred/mycred.php' ) )
			{
				/**
				* Disabled this workaround until we have checked if this issue still exists
				*
				* @since 0.0.1
				*/
			    remove_filter( 'the_content', 'chiedolabs_shortcode_autop' );
			    remove_filter( 'the_excerpt', 'chiedolabs_shortcode_autop' );
			}


			/**
			* Disabled this workaround until we have checked if this issue still exists
			*
			* @since 0.0.1
			*/
			// remove empty p tag which wrap <div>
			//$content = preg_replace( '/<p>(<!--[^>]*-->)*\n*(<div)/s', '$2', $content );
			//$content = preg_replace( '/(<\/div>)(.*)\n*<\/p>/s', '$1$2', $content );

			// Replace each &#038; with &
			$content = str_replace( '&#0f38;', '&', $content );



			return balanceTags( $content );
		}

		/**
		 * Generate shortcode pattern ( for Ig shortcodes only )
		 * @global type $shortcode_tags
		 * @return pattern which contains only shortcodes of ProStyler Builder
		 */
		public static function shortcodes_pattern( $tags = '' ) {

			global $St_Pb_Shortcodes;
			global $shortcode_tags;
			$shortcode_tags_clone = $shortcode_tags;
			$shortcode_tags       = empty( $tags ) ? ( ! empty ( $St_Pb_Shortcodes ) ? $St_Pb_Shortcodes : ST_Pb_Helper_Shortcode::st_pb_shortcode_tags() ) : $tags;
			$pattern              = get_shortcode_regex();
			$shortcode_tags       = $shortcode_tags_clone;

			return "/$pattern/s";
		}

		/**
		 * List of WR shortcode tags, seperating by '|', to use in regular expression
		 *
		 * @global array $St_Pb_Shortcodes
		 * @global string $sc_tag
		 * @return string
		 */
		public static function regex_wr_sc_tag( $sc_tag = '' ) {
			if ( empty( $sc_tag ) ) {
				global $St_Pb_Shortcodes;
				$st_shortcode_tags = ! empty ( $St_Pb_Shortcodes ) ? $St_Pb_Shortcodes : ST_Pb_Helper_Shortcode::st_pb_shortcode_tags();
				$tagnames          = array_keys( $st_shortcode_tags );
			} else {
				$tagnames = (array) $sc_tag;
			}
			$tagregexp         = join( '|', array_map( 'preg_quote', $tagnames ) );

			return $tagregexp;
		}

		/**
		 * Regex to get opening tag & parameters of a shortcode
		 *
		 * @param string $tagregexp The shortcode tags list, seperating by '|'
		 * @return string
		 */
		public static function regex_sc_tag_open( $tagregexp ) {
			// replace opening tag
			$regex   = '\\[' // Opening bracket
			. '(\\[?)' // 1: Optional second opening bracket for escaping shortcodes: [[tag]]
			. "($tagregexp)" // 2: Shortcode name
			. '(?![\\w-])' // Not followed by word character or hyphen
			. '(' // 3: Unroll the loop: Inside the opening shortcode tag
			. '[^\\]\\/]*' // Not a closing bracket or forward slash
			. '(?:'
			. '\\/(?!\\])' // A forward slash not followed by a closing bracket
			. '[^\\]\\/]*' // Not a closing bracket or forward slash
			. ')*?'
			. ')'
			. '(?:'
			. '(\\/)' // 4: Self closing tag ...
			. '\\]' // ... and closing bracket
			. '|'
			. '\\]' // Closing bracket
			. ')'
			. '(\\]?)'; // 6: Optional second closing brocket for escaping shortcodes: [[tag]]

			return $regex;
		}

		/**
		 * Remove all Ig shortcodes from content
		 *
		 * @param string $content The content which contain shortcodes
		 * @param string $sc_tag The shortcode tag want to remove
		 *
		 * @return string Content without shortcode tags
		 */
		public static function remove_wr_shortcodes( $content, $sc_tag = '' ) {
			// ig shortcode list
			$tagregexp = self::regex_wr_sc_tag( $sc_tag );

			// ig shortcode regex
			$regex = self::regex_sc_tag_open( $tagregexp );

			$content = preg_replace( "/$regex/s", '<p>', $content );

			// replace closing tag
			$regex   = "\\[\\/($tagregexp)\\]";
			$content = preg_replace( "/$regex/s", '</p>', $content );

			// remove redundant p tag
			$content = preg_replace( '/(<p>)+/', '<p>', $content );
			$content = preg_replace( '/(<\/p>)+/', '</p>', $content );
			$content = preg_replace( '/(<p>\s*<\/p>)+/', '', $content );

			return $content;
		}

		/**
		 * Split string by regular expression, then replace nodes by string ( [wrapper string]node content[/wrapper string] )
		 *
		 * @param type $pattern
		 * @param type $content
		 * @param type $content_flag
		 * @param type $append_
		 *
		 * @return type string
		 */
		private static function wrap_content( $pattern, $content, $content_flag, $append_ ) {
			$nodes      = preg_split( $pattern, $content, - 1, PREG_SPLIT_OFFSET_CAPTURE );
			$idx_change = 0;
			foreach ( $nodes as $node ) {
				$replace   = $node[0];
				$empty_str = self::check_empty_( $content );
				if ( strlen( trim( $replace ) ) && strlen( trim( $empty_str ) ) ) {
					$offset       = intval( $node[1] ) + $idx_change;
					$replace_html = $replace;

					$content     = substr_replace( $content, str_replace( $content_flag, $replace_html, $append_ ), $offset, strlen( $replace ) );
					$idx_change += strlen( $append_ ) - strlen( $content_flag ) - ( strlen( $replace ) - strlen( $replace_html ) );
				}
			}

			return $content;
		}

		/**
		 * Check if string is empty (no real content)
		 *
		 * @param type $content
		 *
		 * @return type
		 */
		public static function check_empty_( $content ) {
			$content = preg_replace( '/<p[^>]*?>/', '', $content );
			$content = preg_replace( '/<\/p>/', '', $content );
			$content = preg_replace( '/["\']/', '', $content );
			$content = str_replace( '&nbsp;', '', $content );

			return $content;
		}

		/**
		 * Rebuild pagebuilder from Shortcode content
		 *
		 * @param type $content
		 * @param type $column : whether this content is wrapped by a column or not
		 * @param type $refine : true only first time call
		 *
		 * @return type ProStyler Builder content for Admin
		 */
		public static function do_shortcode_admin( $content = '', $column = false, $refine = false ) {
			if ( empty( $content ) ) {
				return '';
			}

			// check if Free Shortcode Plugin is not installed
			global $shortcode_tags;
			if ( ! array_key_exists( 'st_text', $shortcode_tags ) ) {
				return __( 'You have not activated <b>"WR Free Shortcodes"</b> plugin. Please activate it before using PageBuilder.', ST_PBL );
			}


			$content = trim( $content );

			$content_flag = 'X';
			if ( $refine ) {
				// remove duplicator wrapper
				$row_start = '\[st_row';
				$col_start = '\[st_column';
				$row_end   = '\[\/st_row\]';
				$col_end   = '\[\/st_column\]';
				$content   = preg_replace( "/$row_start([^($row_start)|($col_start)]*)$col_start/", '[st_row][st_column', $content );
				$content   = preg_replace( "/$col_end([^($row_end)|($col_end)]*)$row_end/", '[/st_column][/st_row]', $content );

				// wrap alone shortcode ( added in Classic Editor )
				$pattern = self::shortcodes_pattern( array( 'st_row' => '', 'st_column' => '' ) );
				$append_ = "[st_row][st_column]{$content_flag}[/st_column][/st_row]";
				$content = self::wrap_content( $pattern, $content, $content_flag, $append_ );
			}


			// wrap alone text
			self::$pattern = self::shortcodes_pattern();

			$pattern = self::$pattern;
			$append_ = $column ? "[st_text el_title=\"\"]{$content_flag}[/st_text]" : "[st_row][st_column][st_text el_title=\"\"]{$content_flag}[/st_text][/st_column][/st_row]";
			$content = self::wrap_content( $pattern, $content, $content_flag, $append_ );


			return preg_replace_callback( self::$pattern, array( 'self', 'do_shortcode_tag' ), $content );
		}

		public static function do_shortcode_tag( $m ) {

			// allow [[foo]] syntax for escaping a tag
			if ( $m[1] == '[' && $m[6] == ']' ) {
				return substr( $m[0], 1, - 1 );
			}

			$tag     = $m[2];
			$content = isset( $m[5] ) ? trim( $m[5] ) : '';

			return call_user_func( array( 'self', 'shortcode_to_pagebuilder' ), $tag, $content, $m[0], $m[3] );
		}

		/**
		 * Return html structure of shortcode in Page Builder area
		 *
		 * @param type $shortcode_name
		 * @param type $attr
		 * @param type $content
		 */
		public static function shortcode_to_pagebuilder( $shortcode_name, $content = '', $shortcode_data = '', $shortcode_params = '' ) {


			$class = ST_Pb_Helper_Shortcode::get_shortcode_class( $shortcode_name );

			if ( class_exists( $class ) ) {
				global $St_Pb;
				$elements = $St_Pb->get_elements();
				$instance = isset( $elements['element'][strtolower( $class )] ) ? $elements['element'][strtolower( $class )] : null;
				if ( ! is_object( $instance ) ) {
					$instance = new $class();
				}
				$instance->init_element();
				if(isset($instance->config))
					$instance->config['wp_shortcode_name'] = $shortcode_name;

				$el_title = '';
				if ( $class != 'ST_Widget' ) {
					// extract param of shortcode ( now for column )
					if ( isset( $instance->config['extract_param'] ) ) {
						$output = shortcode_parse_atts( $shortcode_params );
						foreach ( $instance->config['extract_param'] as $param ) {
							if ( isset( $output[$param] ) ) {
								$instance->params[$param] = ST_Pb_Utils_Common::remove_quotes( $output[$param] );
							}
						}
					}

					// get content in pagebuilder of shortcode: Element Title must always first option of Content tab
					if ( isset( $instance->items['content'] ) && isset( $instance->items['content'][0] ) ) {
						$title = $instance->items['content'][0];
						if ( $title['role'] == 'title' ) {
							$params   = shortcode_parse_atts( $shortcode_params );
							$el_title = ! empty( $params[$title['id']] ) ? $params[$title['id']] : __( '', ST_PBL );
						}
					}
				} else {
					$widget_info                   = ST_Pb_Helper_Shortcode::extract_widget_params( $shortcode_data );
					$el_title                      = ! empty( $widget_info['title'] ) ? $widget_info['title'] : '';
					$params                        = ST_Pb_Helper_Shortcode::extract_params( $shortcode_data );
					$instance->config['shortcode'] = $params['widget_id'];
					$instance->config['el_type']   = 'widget';
				}

				// ------------------------------------------------------
				//  Effort to solve blank page issue when too many shortcodes
				// ------------------------------------------------------
				@ini_set('memory_limit',  '12800M');
				@ini_set('max_execution_time',  60000);
				@ini_set('max_input_time',  60000);

				$shortcode_view = $instance->element_in_pgbldr( $content, $shortcode_data, $el_title );

				return $shortcode_view[0];
			}
		}

		/**
		 * Extract sub-shortcode content of a shortcode
		 *
		 * @param type $content
		 * @param type $recursion
		 *
		 * @return type
		 */
		public static function extract_sub_shortcode( $content = '', $recursion = false ) {
			if ( empty( self::$pattern ) ) {
				self::$pattern = self::shortcodes_pattern();
			}
			preg_match_all( self::$pattern, $content, $out );
			if ( $recursion ) {
				return self::extract_sub_shortcode( $out[5][0] );
			}

			// categorize sub shortcodes content
			$sub_sc_tags = array();

			// sub sortcodes content
			$sub_sc_data = $out[0];

			foreach ( $sub_sc_data as $sc_data ) {

				// get shortcode name
				preg_match( '/\[([^\s\]]+)/', $sc_data, $matches );
				if ( $matches ) {
					$sc_class                 = self::get_shortcode_class( $matches[1] );
					$sub_sc_tags[$sc_class][] = $sc_data;
				}
			}

			return $sub_sc_tags;
		}

		/**
		 * Merge Shortcode Content & Sub Shortcode Content
		 *
		 * @param type $shortcode_content
		 * @param type $sub_shortcode_content
		 *
		 * @return type
		 */
		public static function merge_shortcode_content( $shortcode_content, $sub_shortcode_content ) {
			if ( empty( self::$pattern ) ) {
				self::$pattern = self::shortcodes_pattern();
			}
			preg_match_all( self::$pattern, $shortcode_content, $out );

			$merge_shortcode                      = array();
			$merge_shortcode['shortcode_tag']     = "[{$out[2][0]}";
			$merge_shortcode['shortcode_params']  = "{$out[3][0]}]";
			$merge_shortcode['shortcode_content'] = $sub_shortcode_content;
			$merge_shortcode['shortcode_tag_end'] = "[/{$out[2][0]}]";
			$merge_shortcode                      = implode( '', $merge_shortcode );

			return stripslashes( $merge_shortcode );
		}

		/**
		 * Extract setting params of Widget Form
		 *
		 * @param type $params
		 *
		 * @return type
		 */
		public static function extract_widget_params( $params ) {
			$params = urldecode( $params );
			$params = preg_replace( '/\[st_widget\s+([A-Za-z0-9_-]+=\"[^"\']*\"\s*)*\s*\]/', '', $params );
			// replace: widget-pages[][title]=Pages 1 => title=Pages 1
			$params = preg_replace( '/([a-z-_])+\[\]\[([^\[\]]+)\]/', '$2', $params );
			$params = str_replace( '[/st_widget]', '', $params );
			parse_str( $params, $instance );

			return $instance;
		}

		/**
		 * Do shortcode & Return final html output for frontend
		 *
		 * @param type $content
		 */
		public static function doshortcode_content( $st_pagebuilder_content ) {
			// remove placeholder text which was inserted to &lt; and &gt;
			$st_pagebuilder_content = ST_Pb_Utils_Placeholder::remove_placeholder( $st_pagebuilder_content, 'wrapper_append', '' );
			$st_pagebuilder_content = preg_replace_callback( '/\[st_widget\s+([A-Za-z0-9_-]+=\"[^"\']*\"\s*)*\s*\](.*)\[\/st_widget\]/Us', array( 'self', 'widget_content' ), $st_pagebuilder_content );

			$output = do_shortcode( $st_pagebuilder_content );

			return $output;
		}

		/**
		 * Replace widget shortcode by Widget output
		 *
		 * @param type $widget_shortcode
		 *
		 * @return type
		 */
		public static function widget_content( $widget_shortcode ) {
			$widget_contents = urldecode( $widget_shortcode[0] );
			// get widget class
			$element = ST_Pb_Helper_Shortcode::extract_params( $widget_contents );
			if ( empty( $element['widget_id'] ) ) {
				return '';
			}
			$widget = $element['widget_id'];
			// get widget settings parameters
			$instance = ST_Pb_Helper_Shortcode::extract_widget_params( $widget_contents );

			// ------------------------------------------------------
			//  YUNUS EDIT
			// ------------------------------------------------------
			$instance_raw = $instance;
			foreach($instance_raw as $key=>$val)
			{
				$instance[ $key ] = stripslashes($val);
			}

			$args     = array( 'widget_id' => strtolower( $widget ) );
			// fix problem of woocommerce
			global $woocommerce;
			if ( isset ( $woocommerce ) && empty ( $woocommerce->query ) ) {
				$woocommerce->query = new WC_Query();
			}

			// init the widget
			$w = new $widget;

			$sidebars_widgets = wp_get_sidebars_widgets();
			// Set a dummy sidebar
			$sidebars_widgets['dummy_wrpb_sidebar'][] = $w->id_base;
			wp_set_sidebars_widgets( $sidebars_widgets );

			// ouput
			ob_start();
			the_widget( $widget, $instance, $args );
			$widget_content = ob_get_clean();

			return $widget_content;
		}

		/**
		 * Render HTML code for shortcode's parameter type
		 * (used in shortcode setting modal)
		 *
		 * @param string $type Type name
		 * @param string $element
		 *
		 * @return string HTML
		 */
		public static function render_parameter( $type, $element = '', $extra_params = null )
		{
			// ------------------------------------------------------
			//  yunus edit
			// ------------------------------------------------------
			if(isset($element['std']) && is_string($element['std']))
			{
				$element['std'] = ST_Pb_Helper_Functions::st_entities_decode($element['std'], true);
			}

			$type_string = self::ucname( $type );
			$class       = 'ST_Pb_Helper_Html_' . $type_string;
			if ( class_exists( $class ) ) {
				$element = ST_Pb_Helper_Type::apply_field_data_source($element);
				return call_user_func( array( $class, 'render' ), $element, $extra_params );
			}

			return false;
		}

		/**
		 * Move this function to a common file
		 *
		 * @param string $string
		 *
		 * @return string
		 */
		public static function ucname( $string ) {
			$string = ucwords( strtolower( $string ) );

			foreach ( array( '-', '\'' ) as $delimiter ) {
				if ( strpos( $string, $delimiter ) !== false ) {
					$string = implode( $delimiter, array_map( 'ucfirst', explode( $delimiter, $string ) ) );
				}
			}

			return $string;
		}

		/**
		 * Method to get only Styling attributes from shortcode content
		 *
		 * @param string $shortcode_name
		 * @param string $shorcode_content
		 *
		 * @return array
		 */
		public static function get_styling_atts( $shortcode_name, $shortcode_content ) {
			// Get the preconfigured styling setting params of shortcode
			$shortcode_class        = self::get_shortcode_class( $shortcode_name );
			$shortcode              = new $shortcode_class;

			// ------------------------------------------------------
			//  yunus edit - was not getting params list
			// ------------------------------------------------------
			$shortcode->element_items();

			$default_styling_params = isset( $shortcode->items['styling'] ) ? $shortcode->items['styling'] : array();
			$styling_atts           = array();


			if ( count( $default_styling_params ) ) {
				// Get inputted params array from shortcode content
				$extracted_params = self::extract_params( $shortcode_content );

				foreach ( $default_styling_params as $param ) {
					if ( $param['id'] && isset ( $extracted_params[$param['id']] ) ) {
						$styling_atts[$param['id']] = $extracted_params[$param['id']];
					}

					// Incase the param has more than 1 values
					// then loop all the values.
					if ( is_array( $param['type'] ) ) {
						foreach ( $param['type'] as $sub_param ) {
							if ( $sub_param['id'] && isset ( $extracted_params[$sub_param['id']] ) ) {
								$styling_atts[$sub_param['id']] = $extracted_params[$sub_param['id']];
							}
						}
					}
				}
			}

			return $styling_atts;
		}



		/**
		 * Checks a field againist its dependency and current value from attributes
		 *
		 * @since 0.0.1
		 */
		static public function field_dependency_check($item, $attrs)
		{
			$dependencies = ! empty( $item['dependency'] ) ? $item['dependency'] : '';
			if ( $dependencies && isset($dependencies[0]) )
			{
				// if not multi dependencies then create multi dep array so loop works normally
				if(! is_array($dependencies[0]))
				{
					$dependencies = array( $dependencies );
				}
			}
			// no dependencies then skip
			else return true;

			// cycle through each dependencies
			foreach($dependencies as $dependency)
			{
				// if depending attr doesnt exist then skip
				if( ! isset($attrs[ $dependency[0] ]) )
				{
					return false;
				}
				// if exists then check againist defined operator and val
				else
				{
					// according to defined operator
					switch($dependency[1])
					{
						case '=':
						case '==':
							if( $dependency[2] == $attrs[ $dependency[0] ] )
							{ }
							else
							{
								return false;
							}
						break;

						case '!=':
						case '=!':
							if( $dependency[2] != $attrs[ $dependency[0] ] )
							{ }
							else
							{
								return false;
							}
						break;

						case '>':
							if( $attrs[ $dependency[0] ] > $dependency[2] )
							{ }
							else
							{
								return false;
							}
						break;

						case '<':
							if( $attrs[ $dependency[0] ] < $dependency[2] )
							{ }
							else
							{
								return false;
							}
						break;
					}
				}
			}

			return true;
		} // end func


		/**
		 * Checks if params array already has id, if not then generate new id and assign to param
		 * @param  array $params attributes of element
		 * @return array         params with id_wrapper
		 */
		static public function check_and_assign_id($params, $name='', $force=false)
		{
			static $log, 
					$i;

			if(! isset($params['id_wrapper'])
				OR ! trim($params['id_wrapper'])
				OR $force===true)
			{
				$params['id_wrapper'] = self::generate_id($params, $name);
			}
			elseif( in_array($params['id_wrapper'], $log) ) // avoid duplicate ids
			{
				$i++;
				$params['id_wrapper'] .= '_'.$i;
			}

			$log[] = $params['id_wrapper'];

			return $params;
		}


		static public function generate_id($params, $name='')
		{
			if($name) return $name.'_'.md5(serialize($params));
			return 'elm_'.uniqid();
		}



	}



}
